#! /bin/sh
#---------------------------------------------------------
# Create and configure QMGR
#
# Arguments:
# $1 : Queue manager name(mandatory)
# $2 : Source mqsc file name(option)
# Read <QMGR>.mqsc.src and <QMGR>.ini from current directory
# Write <QMGR>.mqsc
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#---------------------------------------------------------
# return code
RET_OK=0
RET_NG=1

# start up wait time
QMGR_STARTUP_TIME=5
CMDSVR_STARTUP_TIME=3

#---------------------------------------------
# Functions for Command Execution
#---------------------------------------------
function fnExecCmd
{
  EXECCMD=$1
  printf "\nCommand: $EXECCMD\n\n"
  eval $EXECCMD
  return $?
}

#-----------------------------------------------------
# Functions for Command Execution with Confirmation
#-----------------------------------------------------
function fnExecCmdConfirm
{
  EXECCMD=$1
  printf "\nCommand: $EXECCMD\n\n"

  printf "Are you sure you want to run above command? [y/n]: "
  read ANS
  if [[ $ANS != [Yy] ]]
  then
    printf "Operation canceld.\n\n"
    return $RET_NG
  fi
  eval $EXECCMD
  return $?
}

#---------------------------------------------
# Functions for Creating QMGR
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------
function fnCreateQmgr
{
  dspmq -m $TARGET_QMGR > /dev/null 2>&1
  if (($? == RET_OK))
  then
    printf "\nQMGR(%s) already exist.\n\n" $TARGET_QMGR

    printf "Do you want to continue processing? [y/n]: "
    read ANS
    if [[ $ANS != [Yy] ]]
    then
      printf "Operation canceld.\n\n"
      return $RET_NG
    fi
  else
    fnExecCmdConfirm "crtmqm $TARGET_QMGR"
  fi
  return $?
}

#---------------------------------------------
# Functions for Starting qmgr
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------
function fnStartQmgr
{
  EXECCMD="strmqm $TARGET_QMGR"

  printf "Starting Qmgr: $TARGET_QMGR\n"

  fnExecCmd "$EXECCMD"
  #return $?
  return $RET_OK
}

#------------------------------------------------------------------
# Functions for replacing configuration parameters
#
# input:
# $TARGET_QMGR: Queue Manager Name
# ${TARGET_QMGR}.mqsc.src: MQSC source file
#
# output:
# ${TARGET_QMGR}.mqsc: MQSC file
#------------------------------------------------------------------
function fnReplaceParameters
{
  sourceFile=${TARGET_QMGR}.mqsc.src
  if (($# != 0))
  then
      sourceFile=${1}
  fi

  targetFile=${TARGET_QMGR}.mqsc
  paramFile=${TARGET_QMGR}.ini

  if [[ ! -f $sourceFile ]]
  then
    echo "Cannot find source file $sourcefile"
    return $RET_NG
  fi

  if [[ ! -f $paramFile ]]
  then
    echo "Cannot find parameter file $paramFile"
    return $RET_NG
  fi

  if [[ -f $targetFile ]]
  then
    echo "remove $targetFile"
    rm -f $targetFile
  fi

  if [[ -f ${targetFile}.temp ]]
  then
    echo "remove ${targetFile}.temp"
    rm -f ${targetFile}.temp
  fi

  while read line
  do
    if [[ $line = \#* ]]
    then
      #echo "this is comment line"
      :
    elif [[ -n `echo $line | tr -d ' '` ]]
    then
      #echo "this is data line"
      FOLDER="<"`echo $line | cut -d ',' -f 1`">"
      VALUEP=$(echo $line | cut -d ',' -f 2)
      echo "FOLDER: $FOLDER VALUE: $VALUEP"
      VALUE=$(echo $VALUEP | cut -d ',' -f 2 | sed 's/\//\\\//g')

      if [[ -f $targetFile ]]
      then
        #echo "copy file : ${targetFile}.temp to $targetFile"
        cat $targetFile | sed "s/$FOLDER/$VALUE/" > ${targetFile}.temp
        cp -p ${targetFile}.temp $targetFile
      else
        #echo "copy file : $sourceFile to $targetFile"
        cat $sourceFile | sed "s/$FOLDER/$VALUE/" > $targetFile
      fi
    fi
  done < $paramFile

  if [[ -f ${targetFile}.temp ]]
  then
    rm -f ${targetFile}.temp
  fi

  return $RET_OK
}

#---------------------------------------------------
# Functions for configuring QMGR
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------------
function fnConfigQmgr
{
  fnExecCmd "mqpcf mqsc -qm $TARGET_QMGR -wi 60 -f ./${TARGET_QMGR}.mqsc"
  return $?
}

#---------------------------------------------
# Functions for Checking QMGR Status
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------
function fnChkQmgrStatus
{
  dspmq -m $TARGET_QMGR | grep "Running" > /dev/null 2>&1
  if (($? != RET_OK))
  then
    return $RET_NG
  fi

  return $RET_OK
}

#---------------------------------------------
# Functions for Checking CMD Server Status
#
# input:
# $TARGET_QMGR: Queue Manager Name
#---------------------------------------------
function fnChkCmdSvrStatus
{
  dspmqcsv $TARGET_QMGR | grep "Running" > /dev/null 2>&1
  if (($? != RET_OK))
  then
    return $RET_NG
  fi

  return $RET_OK
}

#-----------------------------------------------------------
# Functions for waiting QMGR and Command Server up
#
# input:
# $TARGET_QMGR: Queue Manager Name
# $QMGR_STARTUP_TIME: Queue Manager startup wait timer
# $CMDSVR_STARTUP_TIME: Command Server startup wait timer
#-----------------------------------------------------------
function fnWaitCSVRUp
{
  fnChkQmgrStatus
  if (($? != RET_OK))
  then
    printf "Waiting queue manager startup %d sec ..." $QMGR_STARTUP_TIME
    sleep $QMGR_STARTUP_TIME
  fi

  fnChkCmdSvrStatus
  if (($? != RET_OK))
  then
    printf "Waiting command server startup %d sec ..." $CMDSVR_STARTUP_TIME
    sleep $CMDSVR_STARTUP_TIME
  fi

  fnChkCmdSvrStatus
  if (($? != RET_OK))
  then
    printf "The command server is not started."
    return $RET_NG
  fi

  return $RET_OK
}

#-------
# Main
#-------
USAGE="usage: $0 <Qmgr> [<source mqsc file>]"

if (($# == 0))
then
  echo $USAGE
  exit $RET_NG
fi

TARGET_QMGR=${1}

# create MQSC source file
fnReplaceParameters ${2}

# Create QMGR
fnCreateQmgr
if (($? != RET_OK))
then
  exit $RET_NG
fi

# Start QNGR
fnStartQmgr
if (($? != RET_OK))
then
  exit $RET_NG
fi

# Wait QMGR and Command Server Up
fnWaitCSVRUp
if (($? != RET_OK))
then
  exit $RET_NG
fi

# Configure QMGR
fnConfigQmgr
if (($? != RET_OK))
then
  exit $RET_NG
fi

