# Windows PwowerShell
#---------------------------------------------------------
#
# RR-1 MQI Local Bindings(Thread)
#
# Parameters:
# $1 : Queue manager name
# $2 : Number of connections
# $3 : Messages
# $4 : Message Bytes
# $5 : Log Directory
# $6 : Workload
# $7 : MQ Constant
# $8 : Data File
#
# Environments(./runpfmts.bat):
# $INCL_CON_TIME : Include MQCONN/MQDISC time in measurement,
#                  0:not include, 1:include
# $RCV_BUFF_SIZE : Receive buffer size
# $CON_RETRY     : The number of connection retry
# $CON_INTERVAL  : Connection retry interval
# $RR1_RequestQ  : Request queue name
# $RR1_ReplyQ    : Reply queue name
#
# History
#
# Date        Create/Update           Description
# 2022.02.21  Pulsar Integration Inc. Initial Version
#
#---------------------------------------------------------
param(
    [string]$TARGET_QMGR = "",
    [int]$Connections = 0,
    [int]$Messages = 0,
    [int]$Bytes = 0,
    [string]$LogDirPre = "",
    [string]$Workload = "",
    [string]$MQ_Constant = "",
    [string]$DataFile = ""
)

# return code
[int]$RET_OK=0
[int]$RET_NG=1

$USAGE = $myInvocation.MyCommand.name + " <Qmgr> <Connections> <Messages> <Bytes> <Log Directory> <Workload> <MQ Constant> <Data File>"

if ($PSBoundParameters.Count -lt 6)
{
    Write-output "" "Usage: $USAGE" "" ""

    $usage_document = @"
    RR-1 MQI Local Bindings(Thread)

    Parameters:
    `$1 : Queue manager name
    `$2 : Number of connections
    `$3 : Number of messages
    `$4 : Message Bytes
    `$5 : Log Directory
    `$6 : Workload
    `$7 : MQ Constant (ex. MQPER_PERSISTENT, MQPER_NOT_PERSISTENT)
    `$8 : Data File (Input message file) (Option)
         If not specified, "<workload name>" is used as the message.

    Environments(./runpfmts.bat):
    `$INCL_CON_TIME : Include MQCONN/MQDISC time in measurement,
                     0:not include, 1:include
    `$RCV_BUFF_SIZE : Receive buffer size
    `$CON_RETRY     : The number of connection retry
    `$CON_INTERVAL  : Connection retry interval
    `$RR1_RequestQ  : Request queue name
    `$RR1_ReplyQ    : Reply queue name

"@

    Write-output $usage_document

    exit $RET_NG
}

# import library
. ./lib/libpfmts.ps1

# source environments for performance test
fnSourceEnvironments

[string]$RR1_RequestQ=${env:RR1_RequestQ}
[string]$RR1_ReplyQ=${env:RR1_ReplyQ}
[int]$CON_RETRY=${env:CON_RETRY}
[int]$CON_INTERVAL=${env:CON_INTERVAL}
[int]$RCV_BUFF_SIZE=${env:RCV_BUFF_SIZE}
[int]$INCL_CON_TIME=${env:INCL_CON_TIME}

$TRM = ""
if ($INCL_CON_TIME -eq 1)
{
    $TRM = "-tr"
}

$dateTime = Get-Date -Format "yyyyMMdd_HHmmss"
$LogDir = "${LogDirPre}\${Workload}_${Bytes}_${Connections}_${TARGET_QMGR}_${dateTime}"

mkdir $LogDir

$RequestLogNamePre = "${LogDir}\req"
$ReplyLogNamePre = "${LogDir}\rep"
$RequestStderrLogNamePre = "${LogDir}\stderr_req"
$ReplyStderrLogNamePre = "${LogDir}\stderr_rep"

$logNameReq = "${RequestLogNamePre}_${dateTime}.log"
$logNameRep = "${ReplyLogNamePre}_${dateTime}.log"
$stderrLogReq = "${RequestStderrLogNamePre}_${Counter}_${dateTime}.log"
$stderrLogRep = "${ReplyStderrLogNamePre}_${Counter}_${dateTime}.log"

Write-Host ""
Write-Host "TARGET_QMGR: $TARGET_QMGR"
Write-Host "Connections: $Connections Messages: $Messages Bytes: $Bytes LogDirPre: $LogDirPre Workload: $Workload MQ_Constant: $MQ_Constant DataFile: $DataFile"

# Responder
# primary side: MQGMO_ACCEPT_TRUNCATED_MSG
# secondary side: -l $Bytes
$ArgumentListRep = "-qm $TARGET_QMGR -q $RR1_RequestQ -oq $RR1_ReplyQ MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -n $Messages -cr $CON_RETRY -wp $CON_INTERVAL -ni $Connections -sz $RCV_BUFF_SIZE $TRM -ss -l $Bytes MQMT_REPLY $MQ_Constant MQPMO_NO_SYNCPOINT -im"

Write-Host "Start Responder Program, Get $RR1_RequestQ, Put $RR1_ReplyQ"
Write-Host "mqpgf $ArgumentListRep"

$WaitProcess = @()

$WaitProcess += Start-Process -FilePath mqpgf.exe -RedirectStandardOutput $logNameRep -RedirectStandardError $stderrLogRep -PassThru -NoNewWindow -ArgumentList $ArgumentListRep

Write-Host ""
Write-Host "Start Requester Program, Put $RR1_RequestQ, Get $RR1_ReplyQ"
if ($DataFile -eq "")
{
    $ArgumentListReq = "-qm $TARGET_QMGR -q $RR1_RequestQ -m $Workload -iq $RR1_ReplyQ -rq $RR1_ReplyQ MQMT_REQUEST $MQ_Constant MQPMO_NO_SYNCPOINT -n $Messages -l $Bytes -cr $CON_RETRY -wp $CON_INTERVAL -ni $Connections $TRM -ss -sz $RCV_BUFF_SIZE MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -mc"
}
else
{
    $ArgumentListReq = "-qm $TARGET_QMGR -q $RR1_RequestQ -f $DataFile -iq $RR1_ReplyQ -rq $RR1_ReplyQ MQMT_REQUEST $MQ_Constant MQPMO_NO_SYNCPOINT -n $Messages -l $Bytes -cr $CON_RETRY -wp $CON_INTERVAL -ni $Connections $TRM -ss -sz $RCV_BUFF_SIZE MQGMO_WAIT MQWI_UNLIMITED MQGMO_NO_SYNCPOINT MQGMO_ACCEPT_TRUNCATED_MSG -mc"
}

Write-Host "mqpgf $ArgumentListReq"

$WaitProcess += Start-Process -FilePath mqpgf.exe -RedirectStandardOutput $logNameReq -RedirectStandardError $stderrLogReq -PassThru -NoNewWindow -ArgumentList $ArgumentListReq

Write-Output ""

$index = 0
for ( ; $index -lt $WaitProcess.count; $index++)
{
    do
    {
        try
        {
            $process = Get-Process -Id $WaitProcess[$index].Id -ErrorAction stop | Format-Table
            Write-Output "The number of the sampling process displayed: $index"
            $process
        }
        catch
        {
            Write-Output "The end of the test command has been detected. Process Number: $index"
            break
        }
        Start-Sleep 5
    }
    while(1)
}

Write-Host ""
fnGetElapsedTime_Thread -logName "$logNameReq" -MESSAGES $Messages -CONNECTIONS $Connections | tee-object ${LogDirPre}/console.log -Append

exit $RET_OK
